-- Das Skript PracticePreMigration.sql dient zur Erzeugung einer leeren Migrations-Datenbank.
--
-- Die Migration erfolgt in mehreren Schritten:
--
-- 1. PracticePreMigration.sql
--    Erzeugt eine leere Migrationsdatenbank. Das DB-Schema ist optimiert fr einen
--    einfachen Datentransfer von der alten Paradox-DB zur Firebird-DB. 
--    Es entspricht nicht dem endgltigen Schema.
--
-- 2. Kopieren der Daten
--    Der ServerMgr kopiert die Daten 1:1 von Paradox nach Firebird.
--    Einige Konflikte mit neuen DB-Constraints werden bereits hier aufgelst.
--    Insbesondere die Eindeutigkeit der Patienten-Karteinummern wird hier garantiert, indem
--    doppelte Karteinummern umbenannt und nicht-vorhandene automatisch generiert werden.
--
-- 3. PracticeDataMigration.sql
--    Fhrt weitere Datenmanipulationen durch, um inkonsistente Daten zu entfernen oder zu korrigieren.
--    Nach Beendigung des Skripts darf es keine Datenstze mehr geben, die zur Verletzung
--    eines DB-Constraints in der neuen Datenbank fhren knnen.
--
-- 4. PracticePostMigration.sql                                      
--    berfhrt das Migrations-Schema in das endgltige Firebird-Schema der Version 3.00.
--    Tabellen, die nur zur Migration dienen, werden gelscht. Constraints werden hinzugefgt.
--

SET SQL DIALECT 3;
SET NAMES NONE;
SET AUTODDL ON;

/******************************************************************************/
/****                               Domains                                ****/
/******************************************************************************/
CREATE DOMAIN NAME AS VARCHAR(256) CHARACTER SET UTF8;
CREATE DOMAIN PREVIEW AS BLOB SUB_TYPE 0 SEGMENT SIZE 100;
CREATE DOMAIN USERNAME AS VARCHAR(64) CHARACTER SET UTF8;
CREATE DOMAIN XML AS BLOB SUB_TYPE 0 SEGMENT SIZE 80;
CREATE DOMAIN DICOM_IS AS INTEGER;
CREATE DOMAIN DICOM_UI AS VARCHAR(64) CHARACTER SET UTF8;
CREATE DOMAIN DICOM_DA AS DATE;
CREATE DOMAIN DICOM_LO AS VARCHAR(64) CHARACTER SET UTF8;
CREATE DOMAIN DICOM_PN AS VARCHAR(256) CHARACTER SET UTF8;
CREATE DOMAIN DICOM_SH AS VARCHAR(16) CHARACTER SET UTF8;
CREATE DOMAIN DICOM_TM AS TIME;

/******************************************************************************/
/****                              Generators                              ****/
/******************************************************************************/

CREATE GENERATOR GEN_PATIENT_ID;
CREATE GENERATOR GEN_VORGANG_ID;
CREATE GENERATOR GEN_WORK_ID;


/******************************************************************************/
/**** Generator GEN_VORGANG_ID wird fr den Primary Key von Tabelle GRUPPE ****/
/**** verwendet. Um Kollisionen mit Ansichten der Demo-DB zu vermeiden,    ****/
/**** starten die IDs der benutzerdefinierten Ansichten bei 1000.          ****/
/******************************************************************************/

SET GENERATOR GEN_VORGANG_ID TO 999;

/******************************************************************************/
/****                          Stored Procedures                           ****/
/******************************************************************************/

SET TERM ^ ; 

CREATE PROCEDURE GET_VORGANGID
RETURNS (
    ID INTEGER)
AS
BEGIN
  ID = GEN_ID(GEN_VORGANG_ID, 1); 
  SUSPEND;
END^

CREATE PROCEDURE VIEWAUSWAHL (
    VORNAME VARCHAR(48) CHARACTER SET UTF8,
    NACHNAME VARCHAR(64) CHARACTER SET UTF8,
    PGEBDAT DATE)
RETURNS (
    KNR VARCHAR(15) CHARACTER SET UTF8,
    PNNAME VARCHAR(48) CHARACTER SET UTF8,
    PVNAME VARCHAR(64) CHARACTER SET UTF8,
    GDATE DATE,
    PLZ VARCHAR(10) CHARACTER SET UTF8,
    ORT VARCHAR(48) CHARACTER SET UTF8,
    STRASSE VARCHAR(48) CHARACTER SET UTF8,
    HIDDEN INTEGER,
    PNR VARCHAR(32) CHARACTER SET UTF8)
AS
BEGIN
  SUSPEND;
END^

CREATE PROCEDURE DELETE_STUDY_IF_EMPTY (UID DICOM_UI) AS
BEGIN
	SUSPEND;
END^

SET TERM ; ^


/******************************************************************************/
/****                                Tables                                ****/
/******************************************************************************/

CREATE TABLE DCMIMG (
    VORGNR                    VARCHAR(32) CHARACTER SET UTF8 NOT NULL,
    IMAGEKIND                 INTEGER NOT NULL,
    SOPINSTANCEUID            VARCHAR(64) CHARACTER SET UTF8,
    STUDYINSTANCEUID          VARCHAR(64) CHARACTER SET UTF8,
    SERIESINSTANCEUID         VARCHAR(64) CHARACTER SET UTF8,
    INSTANCENUMBER            INTEGER,
    SYSTEMID                  VARCHAR(64) CHARACTER SET UTF8,
    SENTAT                    TIMESTAMP,
    COMMITEDAT                TIMESTAMP,
    COMMITEDFROM              VARCHAR(255) CHARACTER SET UTF8,
    SYSTEMINFO                VARCHAR(255),
    SOPCLASSUID               VARCHAR(64) CHARACTER SET UTF8,
    MODALITY                  VARCHAR(8) CHARACTER SET UTF8,
    SERIESDATE                DATE,
    SERIESTIME                TIME,
    SERIESDESCRIPTION         VARCHAR(1024) CHARACTER SET UTF8,
    SERIESNUMBER              INTEGER,
    PERFORMINGPHYSICIAN       VARCHAR(64) CHARACTER SET UTF8,
    SCHEDULEDPROCEDURESTEP    VARCHAR(128) CHARACTER SET UTF8,
    FILESETID                 VARCHAR(64) CHARACTER SET UTF8,
    FILESETUID                VARCHAR(64) CHARACTER SET UTF8,
    COMMITMENTTRANSACTIONUID  VARCHAR(64) CHARACTER SET UTF8,
    COMMITED                  INTEGER,
    PNR                       VARCHAR(32) CHARACTER SET UTF8,
    ACCESSIONNUMBER           VARCHAR(16) CHARACTER SET UTF8,
    STUDYDATE                 DATE,
    STUDYTIME                 TIME,
    REFERRINGPHYSICIAN        VARCHAR(64) CHARACTER SET UTF8,
    STUDYID                   VARCHAR(64) CHARACTER SET UTF8,
    STUDYDESCRIPTION          VARCHAR(240) CHARACTER SET UTF8,
    REFERSOPCLASSUID          VARCHAR(64) CHARACTER SET UTF8,
    REFERSOPINSTANCEUID       VARCHAR(64) CHARACTER SET UTF8
);

CREATE TABLE GRUPPE (
    GID                INTEGER NOT NULL,
    PNR                VARCHAR(32) CHARACTER SET UTF8,
    GRUPPENNAME        NAME,
    LASTCHANGE         TIMESTAMP,
    INFO               VARCHAR(2048) CHARACTER SET UTF8,
    BILDER             XML,
    PREVIEW            PREVIEW,
    LASTCHANGEBY       USERNAME,
    CREATEDAT          TIMESTAMP,
    CREATEDBY          USERNAME,
    STUDYINSTANCEUID   DICOM_UI,
    SOPINSTANCEUID     DICOM_UI,
    SERIESINSTANCEUID  DICOM_UI,
    INSTANCENUMBER     DICOM_IS,
    SERIESNUMBER       DICOM_IS
);

CREATE TABLE IMPORTPAT (
    IPD            INTEGER NOT NULL,
    PNR            VARCHAR(32) CHARACTER SET UTF8 NOT NULL,
    EINTRAG        TIMESTAMP,
    LASTCHANGE     TIMESTAMP,
    HIDDEN         SMALLINT,
    PRAXIS         VARCHAR(32) CHARACTER SET UTF8,
    KNR            VARCHAR(36) CHARACTER SET UTF8,
    PNNAME         VARCHAR(64) CHARACTER SET UTF8,
    PVNAME         VARCHAR(48) CHARACTER SET UTF8,
    SYSTEMID       VARCHAR(32) CHARACTER SET UTF8,
    EXTERNID       VARCHAR(32) CHARACTER SET UTF8,
    CARDIDXID      VARCHAR(32) CHARACTER SET UTF8,
    GDATE          DATE,
    PLZ            VARCHAR(10) CHARACTER SET UTF8,
    ORT            VARCHAR(48) CHARACTER SET UTF8,
    STRASSE        VARCHAR(48) CHARACTER SET UTF8,
    TITEL          VARCHAR(48) CHARACTER SET UTF8,
    TELEFON        VARCHAR(50) CHARACTER SET UTF8,
    PATIENTSTATUS  INTEGER,
    SCHWANGER      CHAR(1),
    SEXPAT         CHAR(1),
    STAMMARZT      VARCHAR(32) CHARACTER SET UTF8
);

CREATE TABLE PATIENT (
    PNR            VARCHAR(32) CHARACTER SET UTF8 NOT NULL,
    LASTCHANGE     TIMESTAMP,
    SERVERSTATE    TIMESTAMP,
    PVERSION       VARCHAR(16) CHARACTER SET UTF8,
    KNR            VARCHAR(36) CHARACTER SET UTF8 NOT NULL,
    HIDDEN         SMALLINT,
    ANREDE         VARCHAR(48) CHARACTER SET UTF8,
    GDATE          DATE,
    INFO           BLOB SUB_TYPE 0 SEGMENT SIZE 4096,
    LAENDERKENN    VARCHAR(3) CHARACTER SET UTF8,
    ORT            VARCHAR(48) CHARACTER SET UTF8,
    PATIENTSTATUS  INTEGER,
    PATIMAGE       BLOB SUB_TYPE 0 SEGMENT SIZE 4096,
    PLZ            VARCHAR(10) CHARACTER SET UTF8,
    PVNAME         VARCHAR(48) CHARACTER SET UTF8,
    PNNAME         VARCHAR(64) CHARACTER SET UTF8,
    NAMENSZUSATZ   VARCHAR(48) CHARACTER SET UTF8,
    SCHWANGER      CHAR(1),
    SEXPAT         CHAR(1),
    MARK           SMALLINT,
    STAMMARZT      VARCHAR(32) CHARACTER SET UTF8,
    STRASSE        VARCHAR(48) CHARACTER SET UTF8,
    SYSTEMID       VARCHAR(32) CHARACTER SET UTF8,
    TITEL          VARCHAR(48) CHARACTER SET UTF8,
    TELEFON        VARCHAR(50) CHARACTER SET UTF8,
    VERSIONX       VARCHAR(16) CHARACTER SET UTF8,
    VERSTAT        VARCHAR(10) CHARACTER SET UTF8,
    WICHTIG        BLOB SUB_TYPE 0 SEGMENT SIZE 4096,
    ZUSATZ1        VARCHAR(32) CHARACTER SET UTF8,
    ZUSATZ2        VARCHAR(32) CHARACTER SET UTF8,
    ZUSATZ3        VARCHAR(32) CHARACTER SET UTF8,
    ZUSATZ4        VARCHAR(32) CHARACTER SET UTF8,
    OPENDENTAID    VARCHAR(36) CHARACTER SET UTF8,
    ANLEGEDATUM    TIMESTAMP
);

CREATE TABLE XRAYVIDEO (
    ITYP           SMALLINT NOT NULL,
    VORGNR         VARCHAR(32) CHARACTER SET UTF8,
    ARCHNR         INTEGER,
    AUFNDATUM      DATE,
    AUFNZEIT       TIME,
    AUSFUEHRENDER  VARCHAR(64) CHARACTER SET UTF8,
    BEFUNDCODE     VARCHAR(2) CHARACTER SET UTF8,
    BEMERK         BLOB SUB_TYPE 0 SEGMENT SIZE 256,
    BILDSTATUS     INTEGER,
    IMGSRCID       VARCHAR(240) CHARACTER SET UTF8,
    KLEINBILD      BLOB SUB_TYPE 0 SEGMENT SIZE 1024,
    LASTACCD       DATE,
    LASTACCT       TIME,
    LASTCHANGE     TIMESTAMP,
    SERVERSTATE    TIMESTAMP,
    OBJCLASS       INTEGER,
    OPENDENTAID    VARCHAR(36) CHARACTER SET UTF8,
    ORGFILE        VARCHAR(36) CHARACTER SET UTF8,
    ORGVORG        VARCHAR(32) CHARACTER SET UTF8,
    PNR            VARCHAR(32) CHARACTER SET UTF8,
    SEKOBJEKTE     BLOB SUB_TYPE 0 SEGMENT SIZE 256,
    SUBTYP         INTEGER,
    SYSTEMID       VARCHAR(32) CHARACTER SET UTF8,
    PVERSION       VARCHAR(16) CHARACTER SET UTF8,
    KVOLT          DOUBLE PRECISION,
    MILLIAMP       DOUBLE PRECISION,
    SCHWANGER      SMALLINT,
    ROEDAUER       DOUBLE PRECISION,
    ROEPLATZ       VARCHAR(48) CHARACTER SET UTF8,
    LASTRADIATION  TIMESTAMP,
    HIDEIMAGE      SMALLINT,
    IMGINFO        BLOB SUB_TYPE 0 SEGMENT SIZE 256
);

CREATE TABLE STUDY (
  UID                 DICOM_UI NOT NULL,
  PNR                 VARCHAR(32) CHARACTER SET UTF8 NOT NULL,
  STUDYID             DICOM_SH,
  STUDYDESCRIPTION    DICOM_LO,
  STUDYDATE           DICOM_DA,
  STUDYTIME           DICOM_TM,
  REFERRINGPHYSICIAN  DICOM_PN,
  ACCESSIONNUMBER     DICOM_SH,
  REFERSOPCLASSUID    DICOM_UI,
  REFERSOPINSTANCEUID DICOM_UI
);

CREATE TABLE PRESENTATIONSTATE (
  UID      VARCHAR(32) CHARACTER SET UTF8 NOT NULL,
  GID      INTEGER,
  PREVIEW  PREVIEW,
  DATA     XML,
  VORGNR   VARCHAR(32) CHARACTER SET UTF8 NOT NULL,
  ITYP     SMALLINT NOT NULL
);

CREATE TABLE VERSIONSTATE (
  VID         INTEGER NOT NULL,
  VERSNR      DOUBLE PRECISION,
  LASTCHANGE  TIMESTAMP
);

CREATE TABLE PATHISTORY (
    PNR      VARCHAR(32) CHARACTER SET UTF8 NOT NULL,
    LOGDATE  TIMESTAMP
);

CREATE TABLE ARCHIVED (
  ARCHNR     INTEGER  NOT NULL,
  PNR        VARCHAR(32) CHARACTER SET UTF8,
  VORGNR     VARCHAR(32) CHARACTER SET UTF8 NOT NULL,
  BART       CHAR(1),
  MEDIENTYP  INTEGER,
  ORGFILE    VARCHAR(36) CHARACTER SET UTF8 ,
  AUFNDATUM  DATE
);

CREATE TABLE ARCHTEMP (
    ARID        INTEGER NOT NULL,
    ARCHNR      INTEGER,
    PNR         VARCHAR(32) CHARACTER SET UTF8,
    PRACTICEID  VARCHAR(32) CHARACTER SET UTF8,
    VORGNR      INTEGER,
    BART        CHAR(1),
    MEDIAID     VARCHAR(25) CHARACTER SET UTF8,
    MEDIENTYP   INTEGER,
    SETNR       INTEGER
);

CREATE TABLE ARCINDEX (
    AID        INTEGER NOT NULL,
    PNR        VARCHAR(32) CHARACTER SET UTF8 NOT NULL,
    ZAEHLER    INTEGER,
    VORGNR     INTEGER,
    BART       CHAR(1),
    AUFNDATUM  DATE,
    ORGFILE    VARCHAR(36) CHARACTER SET UTF8,
    FSIZE      BIGINT
);

CREATE TABLE VIDEO (
  VID           INTEGER NOT NULL PRIMARY KEY,
  ARCHNR        INTEGER,
  AUFNDATUM     DATE,
  AUFNZEIT      TIME,
  AUSFUEHRENDER VARCHAR(64)  CHARACTER SET UTF8 ,
  BEFUNDCODE    VARCHAR(2)   CHARACTER SET UTF8,
  BEMERK        BLOB SUB_TYPE 0 SEGMENT SIZE 256,
  BILDSTATUS    INTEGER,
  IMGINFO       BLOB SUB_TYPE 0 SEGMENT SIZE 256,
  IMGSRCID      VARCHAR(240)  CHARACTER SET UTF8 ,
  KLEINBILD     BLOB SUB_TYPE 0 SEGMENT SIZE 1024,
  LASTACCD      DATE,
  LASTACCT      TIME,
  LASTCHANGE    TIMESTAMP,
  OBJCLASS      INTEGER,
  OPENDENTAID   VARCHAR(36)  CHARACTER SET UTF8 ,
  ORGFILE       VARCHAR(36)  CHARACTER SET UTF8 ,
  ORGVORG       INTEGER,
  PNR           INTEGER,
  SEKOBJEKTE    BLOB SUB_TYPE 0 SEGMENT SIZE 256,
  SUBTYP        INTEGER,
  SYSTEMID      VARCHAR(32)  CHARACTER SET UTF8 ,
  PVERSION      VARCHAR(16)  CHARACTER SET UTF8 ,
  VORGNR        INTEGER
);

CREATE TABLE XRAY (
  XID           INTEGER NOT NULL PRIMARY KEY,
  ARCHNR        INTEGER,
  AUFNDATUM     DATE,
  AUFNZEIT      TIME,
  AUSFUEHRENDER VARCHAR(64)  CHARACTER SET UTF8 ,
  BEFUNDCODE    VARCHAR(2)  CHARACTER SET UTF8 ,
  BEMERK        BLOB SUB_TYPE 0 SEGMENT SIZE 256,
  BILDSTATUS    INTEGER,
  IMGINFO       BLOB SUB_TYPE 0 SEGMENT SIZE 256,
  IMGSRCID      VARCHAR(240) CHARACTER SET UTF8,
  KVOLT         DOUBLE PRECISION,
  MILLIAMP      DOUBLE PRECISION,
  KLEINBILD     BLOB SUB_TYPE 0 SEGMENT SIZE 1024,
  LASTACCD      DATE,
  LASTACCT      TIME,
  LASTCHANGE    TIMESTAMP,
  OBJCLASS      INTEGER,
  OPENDENTAID   VARCHAR(36)  CHARACTER SET UTF8 ,
  ORGFILE       VARCHAR(36)  CHARACTER SET UTF8 ,
  ORGVORG       INTEGER,
  PNR           INTEGER,
  SEKOBJEKTE    BLOB SUB_TYPE 0 SEGMENT SIZE 256,
  SUBTYP        INTEGER,
  SYSTEMID      VARCHAR(32)  CHARACTER SET UTF8 ,
  PVERSION      VARCHAR(16)  CHARACTER SET UTF8 ,
  VORGNR        INTEGER,
  SCHWANGER     SMALLINT,
  ROEDAUER      DOUBLE PRECISION,
  ROEPLATZ      VARCHAR(48)  CHARACTER SET UTF8,
  LASTRADIATION TIMESTAMP
);


/******************************************************************************/
/****                             Primary Keys                             ****/
/******************************************************************************/

ALTER TABLE GRUPPE ADD CONSTRAINT PK_GRUPPE PRIMARY KEY (GID);
ALTER TABLE IMPORTPAT ADD CONSTRAINT PK_IMPORTPAT PRIMARY KEY (IPD);
ALTER TABLE PATIENT ADD CONSTRAINT PK_PATIENT PRIMARY KEY (PNR);
ALTER TABLE PATHISTORY ADD CONSTRAINT PK_PATHISTORY PRIMARY KEY (PNR);
ALTER TABLE PRESENTATIONSTATE ADD CONSTRAINT PK_PRESENTATIONSTATE PRIMARY KEY (UID);
ALTER TABLE STUDY ADD CONSTRAINT PK_STUDY PRIMARY KEY (UID);
ALTER TABLE VERSIONSTATE ADD CONSTRAINT PK_VERSIONSTATE PRIMARY KEY (VID);


/******************************************************************************/
/****                             Foreign Keys                             ****/
/******************************************************************************/


/******************************************************************************/
/****                             Check Constraints                        ****/
/******************************************************************************/

ALTER TABLE GRUPPE ADD CONSTRAINT CHK_DICOMATTRIBS_GRUPPE CHECK((SOPINSTANCEUID IS NULL) OR ((SERIESINSTANCEUID IS NOT NULL) AND (STUDYINSTANCEUID IS NOT NULL) AND (INSTANCENUMBER IS NOT NULL) AND (SERIESNUMBER IS NOT NULL)));


/******************************************************************************/
/****                               Indices                                ****/
/******************************************************************************/

CREATE INDEX IDX_DCMIMG_COMMITED ON DCMIMG (COMMITED);
CREATE INDEX IDX_DCMIMG_SOPINSTANCEUID ON DCMIMG (SOPINSTANCEUID, SOPCLASSUID);
CREATE INDEX IDX_IMPORTPAT_PRAXIS ON IMPORTPAT (PRAXIS, PNR);
CREATE INDEX IDX_PATIENT_PNNAME_ASC ON PATIENT COMPUTED BY (UPPER(PNNAME));
CREATE DESCENDING INDEX IDX_PATIENT_PNNAME_DESC ON PATIENT COMPUTED BY (UPPER(PNNAME));
CREATE UNIQUE INDEX IDX_PATIENT_KNR ON PATIENT (KNR);
CREATE INDEX IDX_PATIENT_HIDDEN ON PATIENT (HIDDEN);
CREATE INDEX IDX_PATIENT_PVNAME ON PATIENT COMPUTED BY (UPPER(PVNAME));
CREATE INDEX IDX_PATIENT_GDATE ON PATIENT (GDATE);
CREATE UNIQUE INDEX IDX_PATIENT_KNR_PNAME ON PATIENT (KNR, PNNAME, PVNAME, GDATE);
CREATE INDEX IDX_XRAYVIDEO_ITYP ON XRAYVIDEO (ITYP, VORGNR);


/******************************************************************************/
/****                         Triggers for tables                          ****/
/******************************************************************************/

SET TERM ^ ;

CREATE TRIGGER GRUPPE_BI0 FOR GRUPPE
ACTIVE BEFORE INSERT POSITION 0
AS
begin
  if (new.gid < 1) then NEW.GID = GEN_ID(GEN_WORK_ID, 1);
end
^

CREATE TRIGGER IMPORTPAT_BI0 FOR IMPORTPAT
ACTIVE BEFORE INSERT OR UPDATE POSITION 0
AS
begin
  NEW.EINTRAG =  CURRENT_TIMESTAMP;
  NEW.ipd = GEN_ID(GEN_WORK_ID, 1);
end
^

CREATE TRIGGER PATIENT_BI0 FOR PATIENT
ACTIVE BEFORE INSERT OR UPDATE POSITION 0
AS
begin
  if ((New.knr = '') OR (New.knr is null))  then
    New.knr = '**' || New.PNr;
  New.anlegedatum = CURRENT_TIMESTAMP;
  if (char_length(NEW.opendentaid)= 0) then NEW.opendentaid = NEW.pnr;
  if (char_length(NEW.systemid)= 0) then NEW.systemid = 'DBSWIN';
end
^

CREATE TRIGGER PATIENT_BI1 FOR PATIENT
ACTIVE BEFORE UPDATE POSITION 0
AS
begin
  NEW.LASTCHANGE = CURRENT_TIMESTAMP;
end
^

CREATE TRIGGER DCMIMG_AD0 FOR DCMIMG ACTIVE
AFTER DELETE POSITION 0 AS
BEGIN
    EXECUTE PROCEDURE DELETE_STUDY_IF_EMPTY(OLD.studyinstanceuid);
END^

CREATE TRIGGER DCMIMG_AU0 FOR DCMIMG ACTIVE
AFTER UPDATE POSITION 0 AS
BEGIN
    IF (NEW.studyinstanceuid <> OLD.studyinstanceuid) THEN
        EXECUTE PROCEDURE DELETE_STUDY_IF_EMPTY(OLD.studyinstanceuid);
END^

CREATE TRIGGER GRUPPE_AD0 FOR GRUPPE ACTIVE
AFTER DELETE POSITION 0 AS
BEGIN
    EXECUTE PROCEDURE DELETE_STUDY_IF_EMPTY(OLD.studyinstanceuid);
END^

CREATE TRIGGER GRUPPE_AU0 FOR GRUPPE ACTIVE
AFTER UPDATE POSITION 0 AS
BEGIN
    IF (NEW.studyinstanceuid <> OLD.studyinstanceuid) THEN
        EXECUTE PROCEDURE DELETE_STUDY_IF_EMPTY(OLD.studyinstanceuid);
END^

CREATE TRIGGER XRAYVIDEO_AU0 FOR XRAYVIDEO ACTIVE
AFTER UPDATE POSITION 0 AS
BEGIN
    IF (NEW.pnr <> OLD.pnr) THEN
        DELETE FROM presentationstate
        WHERE vorgnr = NEW.vorgnr AND ityp = NEW.ityp;
END^

SET TERM ; ^

/******************************************************************************/
/****                          Stored Procedures                           ****/
/******************************************************************************/

SET TERM ^ ;

ALTER PROCEDURE DELETE_STUDY_IF_EMPTY (UID DICOM_UI) AS
BEGIN
    DELETE FROM study s
    WHERE s.uid=:uid
    AND NOT EXISTS (SELECT d.studyinstanceuid FROM dcmimg d WHERE d.studyinstanceuid = s.uid)
    AND NOT EXISTS (SELECT g.studyinstanceuid FROM gruppe g WHERE g.studyinstanceuid = s.uid);
END^

SET TERM ; ^


COMMIT;
