SET SQL DIALECT 3;
SET NAMES NONE;
SET AUTODDL ON;

-- Lsche alle Zeilen aus XRAYVIDEO, die 
-- (a) eine ungltige VORGNR (=NULL) enthalten, oder
-- (b) eine ungltige PNR enthalten (entweder NULL oder ein Wert, der in Tabelle PATIENT nicht existiert).
-- Die referenzierten Bilder knnen ber den DB-Reconstruct Mechanismus wieder importiert werden.
DELETE FROM XRAYVIDEO xv
WHERE xv.VORGNR IS NULL           
OR xv.PNR IS NULL
OR NOT EXISTS (SELECT 1 FROM PATIENT p WHERE p.PNR = xv.PNR);

-- Die Felder (VORGNR, ITYP) sind jetzt eindeutig und ungleich NULL. Wir knnen nun den Primary Key erzeugen.
-- Der zugehrige Index wird zur Performance Steigerung der Statements weiter unten bentigt. Andernfalls knnen
-- die Statements je nach Gre der Datenbank mehrere Stunden bentigen.
UPDATE RDB$RELATION_FIELDS SET RDB$NULL_FLAG = 1 WHERE RDB$FIELD_NAME = 'VORGNR' AND RDB$RELATION_NAME = 'XRAYVIDEO';
COMMIT;
ALTER TABLE XRAYVIDEO ADD CONSTRAINT PK_XRAYVIDEO PRIMARY KEY (VORGNR, ITYP);

-- Temporrer Index zur Performance-Steigerung der Statements weiter unten.
CREATE INDEX IDX_DCMIMG_VORGNR_TEMP ON DCMIMG (VORGNR, IMAGEKIND);
CREATE INDEX IDX_DCMIMG_STUDY_TEMP ON DCMIMG (STUDYINSTANCEUID);

-- Lsche alle Zeilen aus DCMIMG, die sich auf ein Bild beziehen, das nicht mehr existiert.
DELETE FROM DCMIMG d
WHERE NOT EXISTS (SELECT 1 FROM XRAYVIDEO xv
                  WHERE xv.VORGNR = d.VORGNR AND xv.ITYP = d.IMAGEKIND);

-- Lsche alle Zeilen aus DCMIMG, die sich auf dasselbe Bild, aber unterschiedliche 
-- Patienten beziehen. In diesem Fall behalten wir diejenige Zeile, deren PNR mit der PNR aus 
-- Tabelle XRAYVIDEO bereinstimmt und lschen alle anderen. 
-- Weisen alle Zeilen eine falsche PNR auf, dann behalten wir (willkrlich) nur die Zeile 
-- mit der kleinsten PNR.
DELETE FROM DCMIMG d
WHERE EXISTS (SELECT 1 FROM DCMIMG d2, XRAYVIDEO xv
              WHERE xv.VORGNR = d2.VORGNR AND xv.ITYP = d2.IMAGEKIND 
              AND d.VORGNR = d2.VORGNR AND d.IMAGEKIND = d2.IMAGEKIND 
              AND d.PNR <> d2.PNR AND (d2.PNR = xv.PNR OR d.PNR > d2.PNR)); 

-- Verhindere, dass Bilder einem anderen Patienten zugeordnet sind als ihre zugehrige
-- Untersuchung, indem das Feld DCMIMG.STUDYINSTANCEUID auf NULL gesetzt wird (keiner Untersuchung zugeordnet).
UPDATE DCMIMG d 
SET d.STUDYINSTANCEUID = NULL 
WHERE EXISTS (SELECT 1 FROM XRAYVIDEO xv, DCMIMG d2, XRAYVIDEO xv2
              WHERE xv.VORGNR = d.VORGNR AND xv.ITYP = d.IMAGEKIND 
              AND xv2.VORGNR = d2.VORGNR AND xv2.ITYP = d2.IMAGEKIND
              AND d.STUDYINSTANCEUID = d2.STUDYINSTANCEUID
              AND xv.PNR <> xv2.PNR);

-- Stelle sicher, dass Feld DCMIMG.INSTANCENUMBER nicht NULL ist.
-- Da bisher jedes Bild in einer eigenen Serie enthalten ist, kann der Wert einfach auf 1 gesetzt werden.
UPDATE DCMIMG d
SET d.INSTANCENUMBER = 1
WHERE d.INSTANCENUMBER IS NULL;			  
		   
-- Lsche alle Zeilen in GRUPPE, die zu keinem oder einem gelschten Patienten gehren.
DELETE FROM GRUPPE g
WHERE g.PNR IS NULL
OR NOT EXISTS (SELECT p.PNR FROM PATIENT p WHERE p.PNR = g.PNR);

-- Extrahiere Study-Daten aus DCMIMG und fge sie in die STUDY Tabelle ein.
-- Das Statement unten geht davon aus, dass die Kombination (DCMIMG.STUDYINSTANCEUID,
-- XRAYVIDEO.PNR) eindeutig ist. Dieser Zustand wird durch die Statements weiter oben gewhrleistet.
INSERT INTO STUDY (UID,PNR,STUDYID,STUDYDESCRIPTION,
                   STUDYDATE,STUDYTIME,REFERRINGPHYSICIAN,ACCESSIONNUMBER,
                   REFERSOPCLASSUID,REFERSOPINSTANCEUID)
SELECT d.STUDYINSTANCEUID,xv.PNR,SUBSTRING(MAX(d.STUDYID) FROM 1 FOR 16),SUBSTRING(MAX(d.STUDYDESCRIPTION) FROM 1 FOR 64),
       MIN(d.STUDYDATE),MIN(d.STUDYTIME),MAX(d.REFERRINGPHYSICIAN),MIN(d.ACCESSIONNUMBER),
       MIN(d.REFERSOPCLASSUID),MIN(d.REFERSOPINSTANCEUID)
FROM DCMIMG d, XRAYVIDEO xv
WHERE d.STUDYINSTANCEUID IS NOT NULL 
AND xv.VORGNR = d.VORGNR AND xv.ITYP = d.IMAGEKIND 
GROUP BY d.STUDYINSTANCEUID, xv.PNR;

COMMIT;
RECONNECT;

-- Cleanup.
DROP INDEX IDX_DCMIMG_VORGNR_TEMP;
DROP INDEX IDX_DCMIMG_STUDY_TEMP;

COMMIT;
