--------------------------------------------------------------------------------
-- Skript DuerrDBSWIN.sql dient zur Erzeugung einer leeren DBSWIN Praxis-Datenbank.
-- Es wird in den folgenden Szenarien aufgerufen:
--    - Neuinstallation einer leeren Praxis-DB.
--    - Neuinstallation einer Demo-DB.
--------------------------------------------------------------------------------

SET SQL DIALECT 3;

SET NAMES NONE;

SET AUTODDL ON;


/******************************************************************************/
/****                               Domains                                ****/
/******************************************************************************/

CREATE DOMAIN NAME AS VARCHAR(256) CHARACTER SET UTF8;
CREATE DOMAIN PREVIEW AS BLOB SUB_TYPE 0 SEGMENT SIZE 100;
CREATE DOMAIN USERNAME AS VARCHAR(64) CHARACTER SET UTF8;
CREATE DOMAIN XML AS BLOB SUB_TYPE 0 SEGMENT SIZE 80;
CREATE DOMAIN DICOM_IS AS INTEGER;
CREATE DOMAIN DICOM_UI AS VARCHAR(64) CHARACTER SET UTF8;
CREATE DOMAIN DICOM_DA AS DATE;
CREATE DOMAIN DICOM_LO AS VARCHAR(64) CHARACTER SET UTF8;
CREATE DOMAIN DICOM_PN AS VARCHAR(256) CHARACTER SET UTF8;
CREATE DOMAIN DICOM_SH AS VARCHAR(16) CHARACTER SET UTF8;
CREATE DOMAIN DICOM_TM AS TIME;


/******************************************************************************/
/****                              Generators                              ****/
/******************************************************************************/

CREATE GENERATOR GEN_PATIENT_ID;
CREATE GENERATOR GEN_VORGANG_ID;
CREATE GENERATOR GEN_WORK_ID;


/******************************************************************************/
/**** Generator GEN_VORGANG_ID wird fr den Primary Key von Tabelle GRUPPE ****/
/**** verwendet. Um Kollisionen mit Ansichten der Demo-DB zu vermeiden,    ****/
/**** starten die IDs der benutzerdefinierten Ansichten bei 1000.          ****/
/******************************************************************************/
SET GENERATOR GEN_VORGANG_ID TO 999;


SET TERM ^ ; 

/******************************************************************************/
/****                          Stored Procedures                           ****/
/******************************************************************************/

CREATE PROCEDURE GET_VORGANGID
RETURNS (
    ID INTEGER)
AS
BEGIN
  ID = GEN_ID(GEN_VORGANG_ID, 1); 
  SUSPEND;
END^

-- TOBEIMPLEMENTED: Stored Procedure VIEWAUSWAHL wird nicht mehr verwendet. Beim nchsten DB-Update entfernen.
CREATE PROCEDURE VIEWAUSWAHL (
    VORNAME VARCHAR(48) CHARACTER SET UTF8,
    NACHNAME VARCHAR(64) CHARACTER SET UTF8,
    PGEBDAT DATE)
RETURNS (
    KNR VARCHAR(15) CHARACTER SET UTF8,
    PNNAME VARCHAR(48) CHARACTER SET UTF8,
    PVNAME VARCHAR(64) CHARACTER SET UTF8,
    GDATE DATE,
    PLZ VARCHAR(10) CHARACTER SET UTF8,
    ORT VARCHAR(48) CHARACTER SET UTF8,
    STRASSE VARCHAR(48) CHARACTER SET UTF8,
    HIDDEN INTEGER,
    PNR VARCHAR(32) CHARACTER SET UTF8)
AS
BEGIN
  SUSPEND;
END^

CREATE PROCEDURE DELETE_STUDY_IF_EMPTY (UID DICOM_UI) AS
BEGIN
	SUSPEND;
END^

/******************************************************************************/
/****                          DELETE_INDEX                                ****/
/**** Conditional delete of an index only if existing                         */
/**** Usecase : Some customers have manually generated indexes, that would    */
/***  cause a fail of database update ( which would try to create the same    */
/**   indices again) Issue 4704                                               */
/******************************************************************************/
CREATE PROCEDURE DELETE_INDEX (
    INDEXNAME VARCHAR(80))
AS
declare s1 varchar (120) ;
begin
  IF (exists(select 1 from rdb$indices where RDB$INDEX_NAME = :Indexname)) THEN BEGIN
    S1 = 'DROP INDEX ' || :Indexname || ' ;';
    execute statement S1 ;
  end
end^

SET TERM ; ^


/******************************************************************************/
/****                                Tables                                ****/
/******************************************************************************/

CREATE TABLE DCMIMG (
    VORGNR                    VARCHAR(32) CHARACTER SET UTF8 NOT NULL,
    IMAGEKIND                 INTEGER NOT NULL,
    SOPINSTANCEUID            VARCHAR(64) CHARACTER SET UTF8,
    STUDYINSTANCEUID          VARCHAR(64) CHARACTER SET UTF8,
    SERIESINSTANCEUID         VARCHAR(64) CHARACTER SET UTF8,
    INSTANCENUMBER            INTEGER NOT NULL,
    SYSTEMID                  VARCHAR(64) CHARACTER SET UTF8,
    SENTAT                    TIMESTAMP,
    COMMITEDAT                TIMESTAMP,
    COMMITEDFROM              VARCHAR(255) CHARACTER SET UTF8,
    SYSTEMINFO                VARCHAR(255),
    SOPCLASSUID               VARCHAR(64) CHARACTER SET UTF8,
    MODALITY                  VARCHAR(8) CHARACTER SET UTF8,
    SERIESDATE                DATE,
    SERIESTIME                TIME,
    SERIESDESCRIPTION         VARCHAR(1024) CHARACTER SET UTF8,
    SERIESNUMBER              INTEGER,
    PERFORMINGPHYSICIAN       VARCHAR(64) CHARACTER SET UTF8,
    SCHEDULEDPROCEDURESTEP    VARCHAR(128) CHARACTER SET UTF8,
    FILESETID                 VARCHAR(64) CHARACTER SET UTF8,
    FILESETUID                VARCHAR(64) CHARACTER SET UTF8,
    COMMITMENTTRANSACTIONUID  VARCHAR(64) CHARACTER SET UTF8,
    COMMITED                  INTEGER
);

CREATE TABLE GRUPPE (
    GID                INTEGER NOT NULL,
    PNR                VARCHAR(32) CHARACTER SET UTF8 NOT NULL,
    GRUPPENNAME        NAME,
    LASTCHANGE         TIMESTAMP,
    INFO               VARCHAR(2048) CHARACTER SET UTF8,
    BILDER             XML,
    PREVIEW            PREVIEW,
    PREVIEWINVALID     CHAR(1) DEFAULT '0' NOT NULL,
    LASTCHANGEBY       USERNAME,
    CREATEDAT          TIMESTAMP,
    CREATEDBY          USERNAME,
    STUDYINSTANCEUID   DICOM_UI,
    SOPINSTANCEUID     DICOM_UI,
    SERIESINSTANCEUID  DICOM_UI,
    INSTANCENUMBER     DICOM_IS,
    SERIESNUMBER       DICOM_IS
);

CREATE TABLE PATIENT (
    PNR            VARCHAR(32) CHARACTER SET UTF8 NOT NULL,
    LASTCHANGE     TIMESTAMP,
    SERVERSTATE    TIMESTAMP,
    PVERSION       VARCHAR(16) CHARACTER SET UTF8,
    KNR            VARCHAR(64) CHARACTER SET UTF8 NOT NULL,
    HIDDEN         SMALLINT,
    ANREDE         VARCHAR(48) CHARACTER SET UTF8,
    GDATE          DATE,
    INFO           BLOB SUB_TYPE 0 SEGMENT SIZE 4096,
    LAENDERKENN    VARCHAR(3) CHARACTER SET UTF8,
    ORT            VARCHAR(48) CHARACTER SET UTF8,
    PATIENTSTATUS  INTEGER,
    PATIMAGE       BLOB SUB_TYPE 0 SEGMENT SIZE 4096,
    PLZ            VARCHAR(10) CHARACTER SET UTF8,
    PVNAME         VARCHAR(48) CHARACTER SET UTF8,
    PNNAME         VARCHAR(64) CHARACTER SET UTF8,
    NAMENSZUSATZ   VARCHAR(48) CHARACTER SET UTF8,
    SCHWANGER      CHAR(1),
    SEXPAT         CHAR(1),
    MARK           SMALLINT,
    STAMMARZT      VARCHAR(32) CHARACTER SET UTF8,
    STRASSE        VARCHAR(48) CHARACTER SET UTF8,
    SYSTEMID       VARCHAR(32) CHARACTER SET UTF8,
    TITEL          VARCHAR(48) CHARACTER SET UTF8,
    TELEFON        VARCHAR(50) CHARACTER SET UTF8,
    VERSIONX       VARCHAR(16) CHARACTER SET UTF8,
    VERSTAT        VARCHAR(10) CHARACTER SET UTF8,
    WICHTIG        BLOB SUB_TYPE 0 SEGMENT SIZE 4096,
    ZUSATZ1        VARCHAR(32) CHARACTER SET UTF8,
    ZUSATZ2        VARCHAR(32) CHARACTER SET UTF8,
    ZUSATZ3        VARCHAR(32) CHARACTER SET UTF8,
    ZUSATZ4        VARCHAR(32) CHARACTER SET UTF8,
    OPENDENTAID    VARCHAR(36) CHARACTER SET UTF8,
    ANLEGEDATUM    TIMESTAMP,
    PNAMEDICOM     VARCHAR(256) CHARACTER SET UTF8
);

CREATE TABLE XRAYVIDEO (
    ITYP           SMALLINT NOT NULL,
    VORGNR         VARCHAR(32) CHARACTER SET UTF8 NOT NULL,
    ARCHNR         INTEGER,
    AUFNDATUM      DATE,
    AUFNZEIT       TIME,
    AUSFUEHRENDER  VARCHAR(64) CHARACTER SET UTF8,
    BEFUNDCODE     VARCHAR(2) CHARACTER SET UTF8,
    BEMERK         BLOB SUB_TYPE 0 SEGMENT SIZE 256,
    BILDSTATUS     INTEGER,
    IMGSRCID       VARCHAR(240) CHARACTER SET UTF8,
    KLEINBILD      BLOB SUB_TYPE 0 SEGMENT SIZE 1024,
    LASTACCD       DATE,
    LASTACCT       TIME,
    LASTCHANGE     TIMESTAMP,
    SERVERSTATE    TIMESTAMP,
    OBJCLASS       INTEGER,
    OPENDENTAID    VARCHAR(36) CHARACTER SET UTF8,
    ORGFILE        VARCHAR(36) CHARACTER SET UTF8,
    ORGVORG        VARCHAR(32) CHARACTER SET UTF8,
    PNR            VARCHAR(32) CHARACTER SET UTF8 NOT NULL,
    SEKOBJEKTE     BLOB SUB_TYPE 0 SEGMENT SIZE 256,
    SUBTYP         INTEGER,
    SYSTEMID       VARCHAR(32) CHARACTER SET UTF8,
    PVERSION       VARCHAR(16) CHARACTER SET UTF8,
    KVOLT          DOUBLE PRECISION,
    MILLIAMP       DOUBLE PRECISION,
    SCHWANGER      SMALLINT,
    ROEDAUER       DOUBLE PRECISION,
    ROEPLATZ       VARCHAR(48) CHARACTER SET UTF8,
    LASTRADIATION  TIMESTAMP,
    HIDEIMAGE      SMALLINT,
    EXPORTID	    VARCHAR(12) CHARACTER SET UTF8 NOT NULL,
    IMGINFO        BLOB SUB_TYPE 0 SEGMENT SIZE 256,
	BILDBEWERTUNG  SMALLINT DEFAULT 0,
	BENUTZERBILDBEWERTUNG VARCHAR(64) CHARACTER SET UTF8,
	ZEITBILDBEWERTUNG TIMESTAMP,
    DOSEAREAPRODUCT DOUBLE PRECISION 
);

CREATE TABLE STUDY (
  UID                 DICOM_UI NOT NULL,
  PNR                 VARCHAR(32) CHARACTER SET UTF8 NOT NULL,
  STUDYID             DICOM_SH,
  STUDYDESCRIPTION    DICOM_LO,
  STUDYDATE           DICOM_DA,
  STUDYTIME           DICOM_TM,
  REFERRINGPHYSICIAN  DICOM_PN,
  ACCESSIONNUMBER     DICOM_SH,
  REFERSOPCLASSUID    DICOM_UI,
  REFERSOPINSTANCEUID DICOM_UI
);

CREATE TABLE PRESENTATIONSTATE (
  UID      VARCHAR(32) CHARACTER SET UTF8 NOT NULL,
  GID      INTEGER,
  PREVIEW  PREVIEW,
  DATA     XML,
  VORGNR   VARCHAR(32) CHARACTER SET UTF8 NOT NULL,
  ITYP     SMALLINT NOT NULL
);

CREATE TABLE VERSIONSTATE (
  VID         INTEGER NOT NULL,
  VERSNR      DOUBLE PRECISION,
  LASTCHANGE  TIMESTAMP
);

CREATE TABLE PATHISTORY (
    PNR      VARCHAR(32) CHARACTER SET UTF8 NOT NULL,
    LOGDATE  TIMESTAMP
);


/******************************************************************************/
/****                             Primary Keys                             ****/
/******************************************************************************/

ALTER TABLE DCMIMG ADD CONSTRAINT PK_DCMIMG PRIMARY KEY (VORGNR,IMAGEKIND);
ALTER TABLE GRUPPE ADD CONSTRAINT PK_GRUPPE PRIMARY KEY (GID);
ALTER TABLE PATIENT ADD CONSTRAINT PK_PATIENT PRIMARY KEY (PNR);
ALTER TABLE XRAYVIDEO ADD CONSTRAINT PK_XRAYVIDEO PRIMARY KEY (VORGNR, ITYP);
ALTER TABLE PATHISTORY ADD CONSTRAINT PK_PATHISTORY PRIMARY KEY (PNR);
ALTER TABLE PRESENTATIONSTATE ADD CONSTRAINT PK_PRESENTATIONSTATE PRIMARY KEY (UID);
ALTER TABLE STUDY ADD CONSTRAINT PK_STUDY PRIMARY KEY (UID);
ALTER TABLE VERSIONSTATE ADD CONSTRAINT PK_VERSIONSTATE PRIMARY KEY (VID);


/******************************************************************************/
/****                             Foreign Keys                             ****/
/******************************************************************************/

ALTER TABLE XRAYVIDEO ADD CONSTRAINT FK_XRAYVIDEO_PATIENT FOREIGN KEY (PNR) REFERENCES PATIENT (PNR) ON UPDATE CASCADE ON DELETE CASCADE;
ALTER TABLE PRESENTATIONSTATE ADD CONSTRAINT FK_PRESENTATIONSTATE_GRUPPE FOREIGN KEY (GID) REFERENCES GRUPPE (GID) ON UPDATE CASCADE ON DELETE CASCADE;
ALTER TABLE PRESENTATIONSTATE ADD CONSTRAINT FK_PRESENTATIONSTATE_XRAYVIDEO FOREIGN KEY (VORGNR, ITYP) REFERENCES XRAYVIDEO (VORGNR, ITYP) ON UPDATE CASCADE ON DELETE CASCADE;
ALTER TABLE DCMIMG ADD CONSTRAINT FK_DCMIMG_STUDY FOREIGN KEY (STUDYINSTANCEUID) REFERENCES STUDY (UID) ON UPDATE CASCADE ON DELETE CASCADE;
ALTER TABLE DCMIMG ADD CONSTRAINT FK_DCMIMG_XRAYVIDEO FOREIGN KEY (VORGNR, IMAGEKIND) REFERENCES XRAYVIDEO (VORGNR, ITYP) ON UPDATE CASCADE ON DELETE CASCADE;
ALTER TABLE GRUPPE ADD CONSTRAINT FK_GRUPPE_PATIENT FOREIGN KEY (PNR) REFERENCES PATIENT (PNR) ON UPDATE CASCADE ON DELETE CASCADE;
ALTER TABLE GRUPPE ADD CONSTRAINT FK_GRUPPE_STUDY FOREIGN KEY (STUDYINSTANCEUID) REFERENCES STUDY (UID) ON UPDATE CASCADE ON DELETE CASCADE;
ALTER TABLE STUDY ADD CONSTRAINT FK_STUDY_PNR FOREIGN KEY (PNR) REFERENCES PATIENT (PNR) ON UPDATE CASCADE ON DELETE CASCADE;
ALTER TABLE PATHISTORY ADD CONSTRAINT FK_PATHISTORY_PATIENT FOREIGN KEY (PNR) REFERENCES PATIENT (PNR) ON UPDATE CASCADE ON DELETE CASCADE;


/******************************************************************************/
/****                             Check Constraints                        ****/
/******************************************************************************/

ALTER TABLE XRAYVIDEO ADD CONSTRAINT CHK_XRAYVIDEO_ITYP CHECK (ITYP IN (0,1));
ALTER TABLE PRESENTATIONSTATE ADD CONSTRAINT CHK_PRESENTATIONSTATE_ITYP CHECK (ITYP IN (0,1));
ALTER TABLE DCMIMG ADD CONSTRAINT CHK_DCMIMG_IMAGEKIND CHECK (IMAGEKIND IN (0,1));
ALTER TABLE GRUPPE ADD CONSTRAINT CHK_GRUPPE_PREVIEWINVALID CHECK (PREVIEWINVALID IN ('0', '1'));
ALTER TABLE GRUPPE ADD CONSTRAINT CHK_DICOMATTRIBS_GRUPPE CHECK((SOPINSTANCEUID IS NULL) OR ((SERIESINSTANCEUID IS NOT NULL) AND (STUDYINSTANCEUID IS NOT NULL) AND (INSTANCENUMBER IS NOT NULL) AND (SERIESNUMBER IS NOT NULL)));


/******************************************************************************/
/****                               Indices                                ****/
/******************************************************************************/

CREATE INDEX IDX_DCMIMG_COMMITED ON DCMIMG (COMMITED);
CREATE INDEX IDX_DCMIMG_SOPINSTANCEUID ON DCMIMG (SOPINSTANCEUID, SOPCLASSUID);
CREATE INDEX IDX_PATIENT_PNNAME_ASC ON PATIENT COMPUTED BY (UPPER(PNNAME));
CREATE DESCENDING INDEX IDX_PATIENT_PNNAME_DESC ON PATIENT COMPUTED BY (UPPER(PNNAME));
CREATE UNIQUE INDEX IDX_PATIENT_KNR ON PATIENT (KNR);
CREATE INDEX IDX_PATIENT_HIDDEN ON PATIENT (HIDDEN);
CREATE INDEX IDX_PATIENT_PVNAME ON PATIENT COMPUTED BY (UPPER(PVNAME));
CREATE INDEX IDX_PATIENT_GDATE ON PATIENT (GDATE);
CREATE UNIQUE INDEX IDX_PATIENT_KNR_PNAME ON PATIENT (KNR, PNNAME, PVNAME, GDATE);
CREATE INDEX IDX_XRAYVIDEO_ITYP ON XRAYVIDEO (ITYP, VORGNR);
CREATE UNIQUE INDEX IDX_XRAYVIDEO_EXPORTID ON XRAYVIDEO (EXPORTID);
CREATE UNIQUE INDEX IDX_XRAYVIDEO_UPPER_EXPORTID ON XRAYVIDEO COMPUTED BY (UPPER(EXPORTID));
CREATE INDEX IDX_XRAYVIDEO_SYSTEMID ON XRAYVIDEO (OPENDENTAID, SYSTEMID);
CREATE INDEX IDX_PATIENT_COLLATE_ASC ON PATIENT COMPUTED BY (UPPER(PNNAME COLLATE UNICODE_CI_AI));
CREATE DESCENDING INDEX IDX_PATIENT_COLLATE_DESC ON PATIENT COMPUTED BY (UPPER(PNNAME COLLATE UNICODE_CI_AI));
CREATE INDEX IDX_PATIENT_KNR_COLLATE_ASC ON PATIENT COMPUTED BY (UPPER(KNR COLLATE UNICODE_CI_AI));
CREATE DESCENDING INDEX IDX_PATIENT_KNR_COLLATE_DESC ON PATIENT COMPUTED BY (UPPER(KNR COLLATE UNICODE_CI_AI));
CREATE INDEX IDX_PATIENT_PVNAME_COLLATE_ASC ON PATIENT COMPUTED BY (UPPER(PVNAME COLLATE UNICODE_CI_AI));
CREATE DESCENDING INDEX IDX_PATIENT_PVNAME_COLLATE_DESC ON PATIENT COMPUTED BY (UPPER(PVNAME COLLATE UNICODE_CI_AI));

/******************************************************************************/
/****                               Triggers                               ****/
/******************************************************************************/


SET TERM ^ ;


/******************************************************************************/
/****                         Triggers for tables                          ****/
/******************************************************************************/

CREATE TRIGGER GRUPPE_BI0 FOR GRUPPE
ACTIVE BEFORE INSERT POSITION 0
AS
begin
  if (new.gid < 1) then NEW.GID = GEN_ID(GEN_WORK_ID, 1);
end
^

CREATE TRIGGER PATIENT_BI0 FOR PATIENT
ACTIVE BEFORE INSERT OR UPDATE POSITION 0
AS
begin
  if ((New.knr = '') OR (New.knr is null))  then
    New.knr = '**' || New.PNr;
  New.anlegedatum = CURRENT_TIMESTAMP;
  if (char_length(NEW.opendentaid)= 0) then NEW.opendentaid = NEW.pnr;
  if (char_length(NEW.systemid)= 0) then NEW.systemid = 'DBSWIN';
end
^

CREATE TRIGGER PATIENT_BI1 FOR PATIENT
ACTIVE BEFORE UPDATE POSITION 0
AS
begin
  NEW.LASTCHANGE = CURRENT_TIMESTAMP;
end
^

CREATE TRIGGER DCMIMG_AD0 FOR DCMIMG ACTIVE
AFTER DELETE POSITION 0 AS
BEGIN
    EXECUTE PROCEDURE DELETE_STUDY_IF_EMPTY(OLD.studyinstanceuid);
END^

CREATE TRIGGER DCMIMG_AU0 FOR DCMIMG ACTIVE
AFTER UPDATE POSITION 0 AS
BEGIN
    IF (NEW.studyinstanceuid <> OLD.studyinstanceuid) THEN
        EXECUTE PROCEDURE DELETE_STUDY_IF_EMPTY(OLD.studyinstanceuid);
END^

CREATE TRIGGER GRUPPE_AD0 FOR GRUPPE ACTIVE
AFTER DELETE POSITION 0 AS
BEGIN
    EXECUTE PROCEDURE DELETE_STUDY_IF_EMPTY(OLD.studyinstanceuid);
END^

CREATE TRIGGER GRUPPE_AU0 FOR GRUPPE ACTIVE
AFTER UPDATE POSITION 0 AS
BEGIN
    IF (NEW.studyinstanceuid <> OLD.studyinstanceuid) THEN
        EXECUTE PROCEDURE DELETE_STUDY_IF_EMPTY(OLD.studyinstanceuid);
END^

CREATE TRIGGER XRAYVIDEO_AU0 FOR XRAYVIDEO ACTIVE
AFTER UPDATE POSITION 0 AS
BEGIN
    IF (NEW.pnr <> OLD.pnr) THEN
        DELETE FROM presentationstate
        WHERE vorgnr = NEW.vorgnr AND ityp = NEW.ityp;
END^

CREATE TRIGGER PRESENTATIONSTATE_AU0 FOR PRESENTATIONSTATE ACTIVE
AFTER UPDATE OR DELETE POSITION 0 AS
BEGIN
   UPDATE GRUPPE g SET g.PREVIEWINVALID = '1' WHERE g.GID = old.GID;
END^

SET TERM ; ^

/******************************************************************************/
/****                          Stored Procedures                           ****/
/******************************************************************************/

SET TERM ^ ;

ALTER PROCEDURE DELETE_STUDY_IF_EMPTY (UID DICOM_UI) AS
BEGIN
    DELETE FROM study s
    WHERE s.uid=:uid
    AND NOT EXISTS (SELECT d.studyinstanceuid FROM dcmimg d WHERE d.studyinstanceuid = s.uid)
    AND NOT EXISTS (SELECT g.studyinstanceuid FROM gruppe g WHERE g.studyinstanceuid = s.uid);
END^

SET TERM ; ^
